<?php

class WPC_Tools_Admin_Module_Default_Pages extends WPC_Tools_Admin_Module {

  const MODULE_NAME = 'wpc-tools-default-pages';

  protected static $_instance = null;

  /**
   * Ensures only one instance of WPC_Tools_Admin is loaded or can be loaded.
   */
  public static function instance() {

    if ( is_null( self::$_instance ) ) {
      self::$_instance = new self();
    }

    return self::$_instance;
  }

  /**
   * WPC_Tools_Module_Default_Pages constructor.
   */
  public function __construct() {
    $this->settings_title = str_replace( '-', '_', self::MODULE_NAME );
    $this->set_defaults();
    $this->module_hooks();
  }

  /**
   * Load General Module hooks
   */
  private function module_hooks() {
    WPC_Tools::loader()->add_action( 'admin_init', $this, 'settings_init', 3 );
    WPC_Tools::loader()->add_action( 'admin_menu', $this, 'load_menu', 20 );
    WPC_Tools::loader()->add_action( 'edit_form_after_title', $this, 'notice_to_default_pages' );
    WPC_Tools::loader()->add_action( 'display_post_states', $this, 'state_to_default_pages', 10, 2 );

    if ( wpc_is_plugin_active( 'acf' ) ) {
      WPC_Tools::loader()->add_filter( 'acf/location/rule_types', $this, 'acf_rule_types_to_default_pages' );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_values/wpc_dp_custom_post_type', $this, 'wpc_dp_custom_post_type_acf_rule_values' );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_values/wpc_dp_custom_taxonomy', $this, 'wpc_dp_custom_taxonomy_acf_rule_values' );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_values/wpc_dp_custom_page', $this, 'wpc_dp_custom_page_acf_rule_values' );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_values/wpc_dp_wpc_page', $this, 'wpc_dp_wpc_page_acf_rule_values' );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_match/wpc_dp_custom_post_type', $this, 'acf_rule_match_to_default_pages', 10, 3 );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_match/wpc_dp_custom_taxonomy', $this, 'acf_rule_match_to_default_pages', 10, 3 );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_match/wpc_dp_custom_page', $this, 'acf_rule_match_to_default_pages', 10, 3 );
      WPC_Tools::loader()->add_filter( 'acf/location/rule_match/wpc_dp_wpc_page', $this, 'acf_rule_match_to_default_pages', 10, 3 );
    }
  }

  /**
   * Add module menu item on wp-admin
   */
  public function load_menu( $parent_id ) {

    if ( ! empty( $this->get_post_types() ) || ! empty( $this->get_taxonomies() || ! empty( $this->get_custom_pages() ) ) ) {

      $this->menu_id = add_submenu_page(
        WPC_Tools_Admin::MENU_SLUG,
        __( 'Default Pages', WPC_Tools::TEXT_DOMAIN ),
        __( 'Default Pages', WPC_Tools::TEXT_DOMAIN ),
        'manage_options',
        self::MODULE_NAME,
        [ $this, 'settings_html' ]
      );

      add_action( 'load-' . $this->menu_id, [ $this, 'load_assets' ] );

      global $submenu;
      if ( isset( $submenu[ WPC_Tools_Admin::MENU_SLUG ] ) && current_user_can( 'manage_options' ) ) {
        $submenu[ WPC_Tools_Admin::MENU_SLUG ][0][0] = __( 'General', WPC_Tools::TEXT_DOMAIN );
      }
    }
  }

  /**
   * Register all assets required by the admin plugin.
   */
  public function load_assets() {
    add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_styles' ] );
    add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
  }

  /**
   * Register the stylesheets for the admin area.
   */
  public function enqueue_styles() {
    wp_enqueue_style( WPC_Tools::PLUGIN_NAME, WPC_TOOLS_URL . 'admin/css/' . self::MODULE_NAME . '.css', [], WPC_Tools::VERSION, 'all' );
  }

  /**
   * Register the JavaScript for the admin area.
   */
  public function enqueue_scripts() {
    wp_enqueue_script( WPC_Tools::PLUGIN_NAME, WPC_TOOLS_URL . 'admin/js/' . self::MODULE_NAME . '.js', [ 'jquery' ], WPC_Tools::VERSION, false );
  }

  /**
   * Theme page callback
   */
  public function settings_html() {
    ?>
    <div class="wrap">
      <h1>
        WPC Tools: <?= __( 'Default Pages', WPC_Tools::TEXT_DOMAIN ); ?>
      </h1>

      <form method="post" action="options.php">
        <?php
        settings_fields( $this->settings_title );
        do_settings_sections( $this->settings_title );
        submit_button();
        ?>
      </form>
    </div>
    <?php
  }

  /**
   * Register setting fields
   */
  public function settings_init() {

    // Custom Post Types
    if ( ! empty( $post_types = $this->get_post_types() ) ) {

      add_settings_section(
        $this->settings_title . '_cpt_default_page_settings',
        __( 'Custom Post Types Settings', WPC_Tools::TEXT_DOMAIN ),
        '__return_false',
        $this->settings_title
      );

      foreach ( $post_types as $post_type ) {

        if ( ! post_type_exists( $post_type ) ) {
          continue;
        }

        $post_type_obj = get_post_type_object( $post_type );

        add_settings_field(
          'page_for_' . $post_type,
          esc_html__( $post_type_obj->labels->name, WPC_Tools::TEXT_DOMAIN ),
          [ $this, 'control_dropdown_pages' ],
          $this->settings_title,
          $this->settings_title . '_cpt_default_page_settings',
          [
            "field_name"              => 'page_for_' . $post_type,
            "field_id"                => 'page_for_' . $post_type,
            "field_class"             => '',
            "field_show_option_none"  => __( '&mdash; Select &mdash;' ),
            "field_option_none_value" => '0',
          ]
        );

        register_setting(
          $this->settings_title,
          'page_for_' . $post_type,
          [ $this, 'sanitize_settings' ]
        );

        add_settings_field(
          $post_type . '_per_page',
          sprintf( esc_html__( '%s Per Page', WPC_Tools::TEXT_DOMAIN ), $post_type_obj->labels->name ),
          [ $this, 'control_number' ],
          $this->settings_title,
          $this->settings_title . '_cpt_default_page_settings',
          [
            "field_name"        => $post_type . '_per_page',
            "field_id"          => $post_type . '_per_page',
            "field_class"       => '',
            "field_step"        => 1,
            "field_min"         => - 1,
            "field_description" => sprintf( __( 'Use -1 to show all %s.', WPC_Tools::TEXT_DOMAIN ), $post_type_obj->labels->name )
          ]
        );

        register_setting(
          $this->settings_title,
          $post_type . '_per_page',
          [ $this, 'sanitize_settings' ]
        );
      }
    }

    // Built-in & Custom Taxonomies
    if ( ! empty( $taxonomies = $this->get_taxonomies() ) ) {

      add_settings_section(
        $this->settings_title . '_ct_default_page_settings',
        __( 'Built-in & Custom Taxonomies Settings', WPC_Tools::TEXT_DOMAIN ),
        '__return_false',
        $this->settings_title
      );

      foreach ( $taxonomies as $taxonomy ) {

        if ( ! taxonomy_exists( $taxonomy ) ) {
          continue;
        }

        $taxonomy_obj = get_taxonomy( $taxonomy );

        add_settings_field(
          'page_for_' . $taxonomy,
          esc_html__( $taxonomy_obj->labels->name, WPC_Tools::TEXT_DOMAIN ),
          [ $this, 'control_dropdown_pages' ],
          $this->settings_title,
          $this->settings_title . '_ct_default_page_settings',
          [
            "field_name"              => 'page_for_' . $taxonomy,
            "field_id"                => 'page_for_' . $taxonomy,
            "field_class"             => '',
            "field_show_option_none"  => __( '&mdash; Select &mdash;' ),
            "field_option_none_value" => '0',
          ]
        );

        register_setting(
          $this->settings_title,
          'page_for_' . $taxonomy,
          [ $this, 'sanitize_settings' ]
        );
      }
    }

    // Custom Pages
    if ( ! empty( $custom_pages = $this->get_custom_pages() ) ) {

      add_settings_section(
        $this->settings_title . '_custom_default_page_settings',
        __( 'Custom Default Pages Settings', WPC_Tools::TEXT_DOMAIN ),
        '__return_false',
        $this->settings_title
      );

      foreach ( $custom_pages as $name => $title ) {

        add_settings_field(
          'page_for_' . $name,
          $title,
          [ $this, 'control_dropdown_pages' ],
          $this->settings_title,
          $this->settings_title . '_custom_default_page_settings',
          [
            "field_name"              => 'page_for_' . $name,
            "field_id"                => 'page_for_' . $name,
            "field_class"             => '',
            "field_show_option_none"  => __( '&mdash; Select &mdash;' ),
            "field_option_none_value" => '0',
          ]
        );

        register_setting(
          $this->settings_title,
          'page_for_' . $name,
          [ $this, 'sanitize_settings' ]
        );
      }
    }

    // WPC Tools Pages
    if ( ! empty( $wpc_pages = $this->get_wpc_pages() ) ) {

      add_settings_section(
        $this->settings_title . '_wpc_page_settings',
        __( 'WPC Pages Settings', WPC_Tools::TEXT_DOMAIN ),
        '__return_false',
        $this->settings_title
      );

      foreach ( $wpc_pages as $name => $title ) {

        add_settings_field(
          'page_for_wpc_' . $name,
          $title,
          [ $this, 'control_dropdown_pages' ],
          $this->settings_title,
          $this->settings_title . '_wpc_page_settings',
          [
            "field_name"              => 'page_for_wpc_' . $name,
            "field_id"                => 'page_for_wpc_' . $name,
            "field_class"             => '',
            "field_show_option_none"  => __( '&mdash; Select &mdash;' ),
            "field_option_none_value" => '0',
          ]
        );

        register_setting(
          $this->settings_title,
          'page_for_wpc_' . $name,
          [ $this, 'sanitize_settings' ]
        );
      }
    }
  }

  /**
   * Sanitize input fields on save
   */
  public function sanitize_settings( $input_fields ) {

    return esc_attr( $input_fields );
  }

  public function set_defaults() {

  }

  /**
   * Add notice to default pages
   */
  public function notice_to_default_pages() {
    global $post;

    if ( ! empty( $post_types = $this->get_post_types() ) ) {

      foreach ( $post_types as $post_type ) {
        $post_type_obj = get_post_type_object( $post_type );

        if ( wpc_icl_object_id( intval( get_option( 'page_for_' . $post_type ) ), 'page' ) == $post->ID ) {
          echo '<div class="notice notice-warning inline"><p>' . sprintf( __( 'You are currently editing the page that shows your %s.', WPC_Tools::TEXT_DOMAIN ), strtolower( $post_type_obj->labels->name ) ) . '</p></div>';
        }
      }
    }

    if ( ! empty( $taxonomies = $this->get_taxonomies() ) ) {

      foreach ( $taxonomies as $taxonomy ) {
        $taxonomy_obj = get_taxonomy( $taxonomy );

        if ( wpc_icl_object_id( intval( get_option( 'page_for_' . $taxonomy ) ), 'page' ) == $post->ID ) {
          echo '<div class="notice notice-warning inline"><p>' . sprintf( __( 'You are currently editing the page that shows your %s.', WPC_Tools::TEXT_DOMAIN ), strtolower( $taxonomy_obj->labels->name ) ) . '</p></div>';
        }
      }
    }

    if ( ! empty( $custom_pages = $this->get_custom_pages() ) ) {

      foreach ( $custom_pages as $name => $title ) {

        if ( wpc_icl_object_id( intval( get_option( 'page_for_' . $name ) ), 'page' ) == $post->ID ) {
          echo '<div class="notice notice-warning inline"><p>' . sprintf( __( 'You are currently editing the %s page.', WPC_Tools::TEXT_DOMAIN ), strtolower( $title ) ) . '</p></div>';
        }
      }
    }

    if ( ! empty( $wpc_pages = $this->get_wpc_pages() ) ) {

      foreach ( $wpc_pages as $name => $title ) {

        if ( wpc_icl_object_id( intval( get_option( 'page_for_wpc_' . $name ) ), 'page' ) == $post->ID ) {
          echo '<div class="notice notice-warning inline"><p>' . sprintf( __( 'You are currently editing the %s page.', WPC_Tools::TEXT_DOMAIN ), strtolower( $title ) ) . '</p></div>';
        }
      }
    }
  }

  /**
   * Add state to default pages
   *
   * @param $post_states
   * @param $post
   *
   * @return string
   */
  public function state_to_default_pages( $post_states, $post ) {

    if ( ! empty( $post_types = $this->get_post_types() ) ) {

      foreach ( $post_types as $post_type ) {
        $post_type_obj = get_post_type_object( $post_type );

        if ( wpc_icl_object_id( intval( get_option( 'page_for_' . $post_type ) ), 'page' ) === $post->ID ) {
          $post_states[ 'page_for_' . $post_type ] = sprintf( __( '%s Page', WPC_Tools::TEXT_DOMAIN ), $post_type_obj->labels->name );
        }
      }
    }

    if ( ! empty( $taxonomies = $this->get_taxonomies() ) ) {

      foreach ( $taxonomies as $taxonomy ) {
        $taxonomy_obj = get_taxonomy( $taxonomy );

        if ( wpc_icl_object_id( intval( get_option( 'page_for_' . $taxonomy ) ), 'page' ) === $post->ID ) {
          $post_states[ 'page_for_' . $taxonomy ] = sprintf( __( '%s Page', WPC_Tools::TEXT_DOMAIN ), $taxonomy_obj->labels->name );
        }
      }
    }

    if ( ! empty( $custom_pages = $this->get_custom_pages() ) ) {

      foreach ( $custom_pages as $name => $title ) {

        if ( wpc_icl_object_id( intval( get_option( 'page_for_' . $name ) ), 'page' ) === $post->ID ) {
          $post_states[ 'page_for_' . $name ] = sprintf( __( '%s Page', WPC_Tools::TEXT_DOMAIN ), esc_html( $title ) );
        }
      }
    }

    if ( ! empty( $wpc_pages = $this->get_wpc_pages() ) ) {

      foreach ( $wpc_pages as $name => $title ) {

        if ( wpc_icl_object_id( intval( get_option( 'page_for_wpc_' . $name ) ), 'page' ) === $post->ID ) {
          $post_states[ 'page_for_wpc_' . $name ] = sprintf( __( '%s Page', WPC_Tools::TEXT_DOMAIN ), esc_html( $title ) );
        }
      }
    }

    return $post_states;
  }

  /**
   * ACF Rule Types to default pages
   *
   * @param $choices
   *
   * @return mixed
   */
  public function acf_rule_types_to_default_pages( $choices ) {

    if ( ! empty( $this->get_post_types() ) ) {

      $choices['Default Pages']['wpc_dp_custom_post_type'] = __( "Custom Post Type's Default Page", WPC_Tools::TEXT_DOMAIN );
    }

    if ( ! empty( $this->get_taxonomies() ) ) {

      $choices['Default Pages']['wpc_dp_custom_taxonomy'] = __( "Built-in & Custom Taxonomy's Default Page", WPC_Tools::TEXT_DOMAIN );
    }

    if ( ! empty( $this->get_custom_pages() ) ) {

      $choices['Default Pages']['wpc_dp_custom_page'] = __( "Custom Page's Default Page", WPC_Tools::TEXT_DOMAIN );
    }

    if ( ! empty( $this->get_wpc_pages() ) ) {

      $choices['Default Pages']['wpc_dp_wpc_page'] = __( "WPC Page's Default Page", WPC_Tools::TEXT_DOMAIN );
    }

    return $choices;
  }

  /**
   * wpc_dp_custom_post_type ACF rule values
   *
   * @param $choices
   *
   * @return mixed
   */
  public function wpc_dp_custom_post_type_acf_rule_values( $choices ) {

    if ( ! empty( $post_types = $this->get_post_types() ) ) {

      foreach ( $post_types as $post_type ) {

        $post_type_obj = get_post_type_object( $post_type );

        if ( ! empty( get_option( 'page_for_' . $post_type ) ) ) {
          $choices[ 'page_for_' . $post_type ] = $post_type_obj->labels->name;
        }
      }
    }

    return $choices;
  }

  /**
   * wpc_dp_custom_taxonomy ACF rule values
   *
   * @param $choices
   *
   * @return mixed
   */
  public function wpc_dp_custom_taxonomy_acf_rule_values( $choices ) {

    if ( ! empty( $taxonomies = $this->get_taxonomies() ) ) {

      foreach ( $taxonomies as $taxonomy ) {

        $taxonomy_obj = get_taxonomy( $taxonomy );

        $choices[ 'page_for_' . $taxonomy ] = $taxonomy_obj->labels->name;
      }
    }

    return $choices;
  }

  /**
   * wpc_dp_custom_page ACF rule values
   *
   * @param $choices
   *
   * @return mixed
   */
  public function wpc_dp_custom_page_acf_rule_values( $choices ) {

    if ( ! empty( $custom_pages = $this->get_custom_pages() ) ) {

      foreach ( $custom_pages as $name => $title ) {

        $choices[ 'page_for_' . $name ] = $title;
      }
    }

    return $choices;
  }

  /**
   * wpc_dp_wpc_page ACF rule values
   *
   * @param $choices
   *
   * @return mixed
   */
  public function wpc_dp_wpc_page_acf_rule_values( $choices ) {

    if ( ! empty( $wpc_pages = $this->get_wpc_pages() ) ) {

      foreach ( $wpc_pages as $name => $title ) {

        $choices[ 'page_for_wpc_' . $name ] = $title;
      }
    }

    return $choices;
  }

  /**
   * ACF Rule Match to default pages
   *
   * @param $match
   * @param $rule
   * @param $options
   *
   * @return bool
   */
  public function acf_rule_match_to_default_pages( $match, $rule, $options ) {

    if ( empty( $options['post_id'] ) ) {
      return $match;
    }

    $current_page_id  = $options['post_id'];
    $selected_page_id = wpc_icl_object_id( intval( get_option( $rule['value'] ) ), 'page' );

    if ( $rule['operator'] == "==" ) {

      $match = ( $current_page_id == $selected_page_id );
    } elseif ( $rule['operator'] == "!=" ) {

      $match = ( $current_page_id != $selected_page_id );
    }

    return $match;
  }

  /**
   * Get post types to Default Pages
   *
   * @return array
   */
  public function get_post_types() {

    $args = [
      'public'   => true,
      '_builtin' => false,
    ];

    return apply_filters( 'wpc_default_pages_post_types', get_post_types( $args, 'names', 'and' ) );
  }

  /**
   * Get taxonomies to Default Pages
   *
   * @return array
   */
  public function get_taxonomies() {

    $args = [
      'public'  => true,
      'show_ui' => true,
    ];

    return apply_filters( 'wpc_default_pages_taxonomies', get_taxonomies( $args, 'names', 'and' ) );
  }

  /**
   * Get custom pages to Default Pages
   *
   * @return array
   */
  public function get_custom_pages() {

    return apply_filters( 'wpc_default_pages_custom_pages', [] );
  }

  /**
   * Get WPC Tools pages to Default Pages
   *
   * @return array
   */
  public function get_wpc_pages() {

    return apply_filters( 'wpc_default_pages_wpc_pages', [] );
  }
}
