<?php

/*
Plugin Name: Advanced Custom Fields: Currency Field
Plugin URI: http://wpcoders.io
Description: ACF field to add numbers by currency.
Version: 1.0.0
Author: Adam Fischer
Author URI: http://wpcoders.io
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
*/

// exit if accessed directly
if( ! defined( 'ABSPATH' ) ) exit;


// check if class already exists
if( !class_exists('acf_plugin_currency') ) :


class acf_plugin_currency {

    /*
    * The single instance of the class.
    */
    protected static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

	/*
	*  __construct
	*
	*  This function will setup the class functionality
	*
	*  @type	function
	*  @date	17/02/2016
	*  @since	1.0.0
	*
	*  @param	n/a
	*  @return	n/a
	*/

	function __construct() {

        // vars
        $this->settings_title = 'settings';

        $this->settings = array(
            'version'       => '1.0.0',
            'url'           => plugin_dir_url( __FILE__ ),
            'path'          => plugin_dir_path( __FILE__ ),
            'currencies'    => $this->get_currencies(),
            'number_format' => $this->get_number_format_settings()
        );

		// set text domain
		// https://codex.wordpress.org/Function_Reference/load_plugin_textdomain
		load_plugin_textdomain( 'acf-currency', false, plugin_basename( dirname( __FILE__ ) ) . '/lang' );


		// include field
		add_action( 'acf/include_field_types', 	array( $this, 'include_field_types' ) ); // v5

        // options page
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) , 100);
        add_action( 'admin_init', array( $this, 'settings_init' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_settings_assets' ), 100);

        // utility functions
        if ( ! function_exists( 'acf_currency_get_currencies' ) ) {
            function acf_currency_get_currencies() {
                return acf_plugin_currency::instance()->get_currencies();
            }
        }

        if ( ! function_exists( 'acf_currency_get_number_format_settings' ) ) {
            function acf_currency_get_number_format_settings() {
                return acf_plugin_currency::instance()->get_number_format_settings();
            }
        }

        if ( ! function_exists( 'acf_currency_number_format' ) ) {
            function acf_currency_number_format( $number ) {
                $settings = acf_currency_get_number_format_settings();
                return number_format( $number, $settings['decimals'], $settings['dec_point'], $settings['thousands_sep'] );
            }
        }
	}


	/*
	*  include_field_types
	*
	*  This function will include the field type class
	*
	*  @type	function
	*  @date	17/02/2016
	*  @since	1.0.0
	*
	*  @param	$version (int) major ACF version. Defaults to false
	*  @return	n/a
	*/

	function include_field_types( $version = false ) {

		// support empty $version
		if( !$version ) $version = 5;

		// include
		include_once('fields/acf-currency-v' . $version . '.php');

	}


    /*
    * This functions will create options page
    */

    function enqueue_settings_assets() {

        // vars
        $url = $this->settings['url'];
        $version = $this->settings['version'];

        // register & include JS
        wp_register_script( 'acf-input-currency-settings', "{$url}assets/js/settings.js", array( 'jquery', 'jquery-ui-sortable' ), $version );
        wp_enqueue_script( 'acf-input-currency-settings' );

        // register & include CSS
        wp_register_style( 'acf-input-currency-settings', "{$url}assets/css/settings.css", false, $version );
        wp_enqueue_style( 'acf-input-currency-settings' );
    }

    function add_admin_menu() {

        add_submenu_page(
            'edit.php?post_type=acf-field-group',
            __( 'Currency Field Settings', 'acf-currency' ),
            __( 'Currency', 'acf-currency' ),
            'manage_options',
            'acf-currency-settings',
            array( $this, 'options_page' )
        );

    }

    function settings_init() {

        register_setting(
            $this->settings_title,
            $this->settings_title,
            [ $this, 'sanitize_settings' ]
        );

        add_settings_section(
            $this->settings_title . '_currencies',
            '',
            '__return_false',
            $this->settings_title
        );

        add_settings_field(
            $this->settings_title . '_currencies',
            __( 'Currencies', 'acf-currency' ),
            array( $this, 'currencies_field_render' ),
            $this->settings_title,
            $this->settings_title . '_currencies',
            array(
                "field_name"  => $this->settings_title . '[currencies]',
                "setting_key" => 'currencies',
            )
        );

        add_settings_section(
            $this->settings_title . '_number_format',
            __( 'Number Format', 'acf-currency' ),
            '__return_false',
            $this->settings_title
        );

        add_settings_field(
            $this->settings_title . '_decimals',
            __( 'Decimals', 'acf-currency' ),
            array( $this, 'number_field_render' ),
            $this->settings_title,
            $this->settings_title . '_number_format',
            array(
                "field_name"  => $this->settings_title . '[decimals]',
                "field_id"    => $this->settings_title . '_decimals',
                "setting_key" => 'decimals',
                "field_class" => 'small-text',
                "default"     => 0,
                "description" => __( 'Sets the number of decimal points.', 'acf-currency' )
            )
        );

        add_settings_field(
            $this->settings_title . '_dec_point',
            __( 'Decimal Point', 'acf-currency' ),
            array( $this, 'text_field_render' ),
            $this->settings_title,
            $this->settings_title . '_number_format',
            array(
                "field_name"  => $this->settings_title . '[dec_point]',
                "field_id"    => $this->settings_title . '_dec_point',
                "setting_key" => 'dec_point',
                "field_class" => 'small-text',
                "default"     => '.',
                "description" => __( 'Sets the separator for the decimal point.', 'acf-currency' )
            )
        );

        add_settings_field(
            $this->settings_title . '_thousands_sep',
            __( 'Thousands Separator', 'acf-currency' ),
            array( $this, 'text_field_render' ),
            $this->settings_title,
            $this->settings_title . '_number_format',
            array(
                "field_name"  => $this->settings_title . '[thousands_sep]',
                "field_id"    => $this->settings_title . '_thousands_sep',
                "setting_key" => 'thousands_sep',
                "field_class" => 'small-text',
                "default"     => ',',
                "description" => __( 'Sets the thousands separator.', 'acf-currency' )
            )
        );

    }

    function sanitize_settings( $input_fields ) {

        $valid_fields = array();

        foreach ( $input_fields as $key => $val ) {

            // Currencies
            if ( $key === 'currencies' ) {
                $valid_fields[ $key ] = array();

                if ( ! empty ( $currencies = $input_fields[ $key ] ) ) {
                    foreach ( $currencies as $currency ) {
                        if ( ! empty( $currency['code'] ) && ! empty( $currency['sign'] ) ) {
                            $valid_fields[ $key ][] = $currency;
                        }
                    }
                }

                $valid_fields[ $key ] = json_encode( $valid_fields[ $key ] );
            }

            // Text
            switch ( $key ) {
                case 'dec_point':
                case 'thousands_sep':
                    $valid_fields[ $key ] = strip_tags( stripslashes( trim( $val, '\t\n\r\0\x0B' ) ) );
                    break;
            }

            // Number
            switch ( $key ) {
                case 'decimals':
                    $valid_fields[ $key ] = absint( $val );
                    break;
            }
        }

        return $valid_fields;
    }

    function currencies_field_render( $args ) {

        $field_value = $this->get_setting( $args['setting_key'] );
        ?>

        <div class="currency-wrap <?= empty( $field_value ) ? 'hidden' : ''; ?>">
            <p class="currency-container">
                <label><?= __( 'Code', 'acf-currency' ); ?></label>
                <input type="text" class="small-text" name="<?= $args['field_name'] ?>[0][code]"
                       value="<?= empty( $field_value[0]['code'] ) ? '' : $field_value[0]['code']; ?>">

                <label><?= __( 'Sign', 'acf-currency' ); ?></label>
                <input type="text" class="small-text" name="<?= $args['field_name'] ?>[0][sign]"
                       value="<?= empty( $field_value[0]['sign'] ) ? '' : $field_value[0]['sign']; ?>">

                <a href="#" class="remove-currency"><?= __( 'Remove', 'acf-currency' ); ?></a>
            </p>

            <?php if ( ! empty( $field_value[1] ) ):
                foreach ( array_slice( $field_value, 1 ) as $key => $field_value ): $key ++; ?>
                    <p class="currency-container">
                        <label><?= __( 'Code', 'acf-currency' ); ?></label>
                        <input type="text" class="small-text" name="<?= $args['field_name'] ?>[<?= $key; ?>][code]"
                               value="<?= empty( $field_value['code'] ) ? '' : $field_value['code']; ?>">

                        <label><?= __( 'Sign', 'acf-currency' ); ?></label>
                        <input type="text" class="small-text" name="<?= $args['field_name'] ?>[<?= $key; ?>][sign]"
                               value="<?= empty( $field_value['sign'] ) ? '' : $field_value['sign']; ?>">

                        <a href="#" class="remove-currency"><?= __( 'Remove', 'acf-currency' ); ?></a>
                    </p>
                <?php endforeach;
            endif; ?>
        </div>

        <div class="add-currency-button-wrap">
            <button type="button" name="add_currency" id="add_currency" class="button button-secondary">
                <?= __( 'Add Currency', 'acf-currency' ); ?>
            </button>
        </div>
        <?php
    }

    function text_field_render( $args ) {

        $field_value = $this->get_setting( $args['setting_key'] );

        if ( empty( $field_value ) ) {
            $field_value = $args['default'];
        }

        ?>
        <input type="text" class="<?= $args['field_class']; ?>" id="<?= $args['field_id']; ?>"
               name="<?= $args['field_name'] ?>" value="<?= esc_attr( $field_value ); ?>">
        <?php

        if ( isset( $args['description'] ) ):
            ?>
            <p class="description"><?= $args['description']; ?></p>
            <?php
        endif;

    }

    function number_field_render( $args ) {

        $field_value = $this->get_setting($args['setting_key']);

        if ( empty( $field_value ) ) {
            $field_value = $args['default'];
        }

        ?>
        <input type="number" value="<?= esc_attr( $field_value ); ?>"
               id="<?= $args['field_id']; ?>" class="<?= $args['field_class']; ?>" name="<?= $args['field_name']; ?>">
        <?php

        if ( isset( $args['description'] ) ):
            ?>
            <p class="description"><?= $args['description']; ?></p>
            <?php
        endif;

    }

    function options_page() {

        ?>

        <div class="wrap">

            <h1><?= __( 'Currency Field Settings', 'acf-currency' ); ?></h1>

            <form action='options.php' method='post'>

                <?php
                settings_fields( $this->settings_title );
                do_settings_sections( $this->settings_title );
                submit_button();
                ?>

            </form>

        </div>

        <?php

    }


    /*
    * Get plugin setting value
    */

    function get_setting( $setting_key ) {

        $settings = get_option( $this->settings_title );

        if ( isset( $settings[ $setting_key ] ) ) {

            $json_array = json_decode( $settings[ $setting_key ], true );

            if ( is_array( $json_array ) ) {
                return $json_array;
            }

            return $settings[ $setting_key ];
        }

        return false;
    }

    /*
    * Get currencies
    */
    function get_currencies() {

        return $this->get_setting( 'currencies' );
    }

    /*
    * Get number format settings
    */
    function get_number_format_settings() {

        return array(
            'decimals'      => $this->get_setting( 'decimals' ),
            'dec_point'     => $this->get_setting( 'dec_point' ),
            'thousands_sep' => $this->get_setting( 'thousands_sep')
        );
    }
}


// initialize
new acf_plugin_currency();

// class_exists check
endif;

?>
