<?php

/**
 * Class WPML_TM_Notification_Batch_Template
 */
class WPML_TM_Batch_Report_Email_Template {

	const JOBS_TEMPLATE   = 'email-job-pairs.twig';
	const FOOTER_TEMPLATE = 'email-footer.twig';
	const HEADER_TEMPLATE = 'email-header.twig';

	/**
	 * @var WPML_Twig_Template
	 */
	private $template_service;

	/**
	 * @var WPML_TM_Blog_Translators
	 */
	private $blog_translators;

	/**
	 * @var SitePress
	 */
	private $sitepress;

	/**
	 * @var array
	 */
	private $assigned_jobs;

	/**
	 * WPML_TM_Batch_Report_Email_Template constructor.
	 *
	 * @param WPML_Twig_Template $template_service
	 * @param WPML_TM_Blog_Translators $blog_translators
	 * @param SitePress $sitepress
	 */
	public function __construct(
		WPML_Twig_Template $template_service,
		WPML_TM_Blog_Translators $blog_translators,
		SitePress $sitepress ) {

		$this->template_service = $template_service;
		$this->blog_translators = $blog_translators;
		$this->sitepress        = $sitepress;
	}

	/**
	 * @param array $language_pairs
	 * @param int $translator_id
	 * @param string $title
	 *
	 * @return null|string
	 */
	public function render_jobs_list( $language_pairs, $translator_id, $title ) {

		$this->empty_assigned_jobs();

		$model = array(
			'strings' => array(
				'strings_text' => __( 'Strings', 'wpml-translation-management' ),
				'start_translating_text' => __( 'start translating', 'wpml-translation-management' ),
				'take' => _x( 'take it', 'Take a translation job waiting for a translator', 'wpml-translation-management' ),
				'strings_link' => admin_url(
					'admin.php?page=wpml-string-translation%2Fmenu%2Fstring-translation.php'
				),
			),
		);

		foreach ( $language_pairs as $lang_pair => $elements ) {

			$languages = explode( '|', $lang_pair );
			$args      = array(
				'lang_from' => $languages[0],
				'lang_to' => $languages[1]
			);

			if ( $this->blog_translators->is_translator( $translator_id, $args ) &&
				 WPML_User_Jobs_Notification_Settings::is_new_job_notification_enabled( $translator_id ) ) {

				$model_elements = array();
				$string_added   = false;

				foreach ( $elements as $element ) {

					if ( ! $string_added || 'string' !== $element['type'] ) {
						$model_elements[] = array(
							'original_link'          => get_permalink( $element['element_id'] ),
							'original_text'          => sprintf( __( 'Link to original document %d', 'wpml-translation-management' ), $element['element_id'] ),
							'start_translating_link' => admin_url(
								'admin.php?page=' . WPML_TM_FOLDER . '%2Fmenu%2Ftranslations-queue.php'
							),
							'type' => $element['type'],
						);

						if ( 'string' === $element['type'] ) {
							$string_added = true;
						}
					}

					$this->add_assigned_job( $element['job_id'], $element['type'] );
				}

				$source_lang                     = $this->sitepress->get_language_details( $languages[0] );
				$target_lang                     = $this->sitepress->get_language_details( $languages[1] );
				$model['lang_pairs'][$lang_pair] = $model_elements;
				$model['lang_pairs'][$lang_pair] = array(
					'title'    => sprintf( __( 'From %s to %s:', 'wpml-translation-management' ), $source_lang['english_name'], $target_lang['english_name'] ),
					'elements' => $model_elements,
				);
			}
		}

		$jobs_count_text = $this->get_jobs_count_text( count( $this->get_assigned_jobs() ) );
		$model['strings']['title'] = sprintf( $title, $jobs_count_text );

		return count( $this->get_assigned_jobs() ) ? $this->template_service->show( $model, self::JOBS_TEMPLATE ) : null;
	}

	public function render_footer(){
		$site_url = get_bloginfo( 'url' );

		$model = array(
			'strings' => array(
				'bottom_text' =>
					__( 'You are receiving this email because you have a translator 
						 account in %1$s. To stop receiving notifications, 
						 log-in to %2$s and unselect "Send me a notification email 
						 when there is something new to translate". Please note that 
						 this will take you out of the translators pool.', 'wpml-translation-management' ),
				'site_url' => $site_url,
				'profile_url' => admin_url( 'profile.php' ),
				'profile_text' => 'your profile',
				'wpml_footer' => sprintf( __( 'Generated by WPML plugin, running on %s.', 'wpml-translation-management' ), $site_url ),
			)
		);

		return $this->template_service->show( $model, self::FOOTER_TEMPLATE );
	}

	/**
	 * @return string
	 */
	public function render_header( $user_name ) {
		$model = array(
			'strings' => array(
				'user_name' => $user_name,
			),
		);

		return $this->template_service->show( $model, self::HEADER_TEMPLATE );
	}

	/**
	 * @param int $jobs_count
	 *
	 * @return string
	 */
	private function get_jobs_count_text( $jobs_count ) {
		$jobs_count_text = ' is 1 job';
		if ( 1 < $jobs_count ) {
			$jobs_count_text = ' are ' . $jobs_count . ' jobs';
		}

		return $jobs_count_text;
	}

	/**
	 * @param int $job_id
	 * @param string $type
	 */
	private function add_assigned_job( $job_id, $type ) {
		$this->assigned_jobs[] = array(
			'job_id' => $job_id,
			'type'   => $type,
		);
	}

	/**
	 * @return array
	 */
	public function get_assigned_jobs() {
		$string_counted = false;
		foreach ( $this->assigned_jobs as $key => $assigned_job ) {
			if ( 'string' === $assigned_job['type'] ) {
				if ( $string_counted ) {
					unset( $this->assigned_jobs[$key] );
				}
				$string_counted = true;
			}
		}

		return $this->assigned_jobs;
	}

	private function empty_assigned_jobs() {
		$this->assigned_jobs = array();
	}
}