<?php

namespace App;

use Roots\Sage\Container;

/**
 * Get the sage container.
 *
 * @param string $abstract
 * @param array $parameters
 * @param Container $container
 *
 * @return Container|mixed
 */
function sage( $abstract = null, $parameters = [], Container $container = null ) {
    $container = $container ?: Container::getInstance();
    if ( ! $abstract ) {
        return $container;
    }

    return $container->bound( $abstract )
        ? $container->makeWith( $abstract, $parameters )
        : $container->makeWith( "sage.{$abstract}", $parameters );
}

/**
 * Get / set the specified configuration value.
 *
 * If an array is passed as the key, we will assume you want to set an array of values.
 *
 * @param array|string $key
 * @param mixed $default
 *
 * @return mixed|\Roots\Sage\Config
 * @copyright Taylor Otwell
 * @link https://github.com/laravel/framework/blob/c0970285/src/Illuminate/Foundation/helpers.php#L254-L265
 */
function config( $key = null, $default = null ) {
    if ( is_null( $key ) ) {
        return sage( 'config' );
    }
    if ( is_array( $key ) ) {
        return sage( 'config' )->set( $key );
    }

    return sage( 'config' )->get( $key, $default );
}

/**
 * @param string $file
 * @param array $data
 *
 * @return string
 */
function template( $file, $data = [] ) {
    if ( remove_action( 'wp_head', 'wp_enqueue_scripts', 1 ) ) {
        wp_enqueue_scripts();
    }

    return sage( 'blade' )->render( $file, $data );
}

/**
 * Retrieve path to a compiled blade view
 *
 * @param $file
 * @param array $data
 *
 * @return string
 */
function template_path( $file, $data = [] ) {
    return sage( 'blade' )->compiledPath( $file, $data );
}

/**
 * @param $asset
 *
 * @return string
 */
function asset_path( $asset ) {
    return sage( 'assets' )->getUri( $asset );
}

/**
 * @param string|string[] $templates Possible template files
 *
 * @return array
 */
function filter_templates( $templates ) {
    $paths         = apply_filters( 'sage/filter_templates/paths', [
        'views',
        'resources/views'
    ] );
    $paths_pattern = "#^(" . implode( '|', $paths ) . ")/#";

    return collect( $templates )
        ->map( function ( $template ) use ( $paths_pattern ) {
            /** Remove .blade.php/.blade/.php from template names */
            $template = preg_replace( '#\.(blade\.?)?(php)?$#', '', ltrim( $template ) );

            /** Remove partial $paths from the beginning of template names */
            if ( strpos( $template, '/' ) ) {
                $template = preg_replace( $paths_pattern, '', $template );
            }

            return $template;
        } )
        ->flatMap( function ( $template ) use ( $paths ) {
            return collect( $paths )
                ->flatMap( function ( $path ) use ( $template ) {
                    return [
                        "{$path}/{$template}.blade.php",
                        "{$path}/{$template}.php",
                        "{$template}.blade.php",
                        "{$template}.php",
                    ];
                } );
        } )
        ->filter()
        ->unique()
        ->all();
}

/**
 * @param string|string[] $templates Relative path to possible template files
 *
 * @return string Location of the template
 */
function locate_template( $templates ) {
    return \locate_template( filter_templates( $templates ) );
}

/**
 * Determine whether to show the sidebar
 *
 * @return bool
 */
function display_sidebar() {
    static $display;
    isset( $display ) || $display = apply_filters( 'sage/display_sidebar', false );

    return $display;
}

/**
 * Page titles
 *
 * @return string
 */
function title( $post_id = null ) {
    if ( empty( $post_id ) ) {
        $post_id = wpc_get_the_id();
    }

    if ( is_home() ) {
        if ( $home = get_option( 'page_for_posts', true ) ) {
            return get_the_title( $home );
        }

        return __( 'Latest Posts', 'hux' );
    }
    if ( is_archive() ) {
        return get_the_archive_title();
    }
    if ( is_search() ) {
        return sprintf( __( 'Search Results for %s', 'hux' ), get_search_query() );
    }
    if ( is_404() ) {
        return __( 'Not Found', 'hux' );
    }

    return get_the_title( $post_id );
}

/**
 * Prints the button markup
 *
 * @param array $args
 *
 * @return string|void
 */
function the_wpc_button( $args ) {

    if ( empty( $args['data'] ) || ( 'link' === $args['data']['link_type'] && empty( $args['data'][ $args['data']['link_type'] . '_link' ] ) ) ) {
        return;
    }

    $defaults = array(
        'data'    => null,
        'classes' => [ 'btn' ]
    );

    $args = wp_parse_args( $args, $defaults );

    $href                               = $args['data'][ $args['data']['link_type'] . '_link' ];
    $label                              = ! empty ( $args['data']['label'] ) ? $args['data']['label'] : '';
    $args['classes'][]                  = 'type-' . $args['data']['link_type'];
    $args['data']['open_in_new_window'] = false;

    if ( 'link' === $args['data']['link_type'] ) {
        $args['data']['open_in_new_window'] = ! empty( $href['target'] );
        $label                              = $href['title'];
        $href                               = $href['url'];
    }

    if ( ! empty ( $label ) ) {
        $args['classes'][] = 'has-label';
    }

    if ( 'anchor' === $args['data']['link_type'] ) {
        $args['classes'][] = 'scroll-to';
    }

    if ( ! empty( $args['data']['link_style'] ) ) {
        $args['classes'][] = $args['data']['link_style'];
    } else {
        $args['classes'][] = 'btn-nostyle';
    }

    if ( ! empty ( $args['data']['icon'] ) ) {
        $args['classes'][] = 'has-icon';
    }
    ?>

    <a class="<?= implode( ' ', $args['classes'] ); ?>"
       href="<?= $href; ?>"
       title="<?= $label; ?>"
       <?php if ( 'file' === $args['data']['link_type'] ) : ?>download<?php endif; ?>
       <?php if ( $args['data']['open_in_new_window'] ) : ?>target="_blank" rel="nofollow"<?php endif; ?>>

        <?php if ( ! empty( $args['data']['icon'] ) ) : ?>
            <i class="<?= $args['data']['icon']; ?> mr-1"></i>
        <?php endif; ?>

        <?= $label; ?>
    </a>
    <?php
}

/**
 * Retrieve the button link
 *
 * @param array $args
 *
 * @return string
 */
function get_the_wpc_button_link( $args ) {
    $link = $args[ $args['link_type'] . '_link' ];

    if ( 'link' === $args['link_type'] ) {
        $link = $link['url'];
    }

    return $link;
}

/**
 * Retrieve responsive image markup
 *
 * @param $attachment_id
 * @param string $size
 * @param null $class
 * @param null $sizes
 *
 * @return string
 */
function get_responsive_attachment( $attachment_id, $size = 'medium', $class = null, $sizes = null ) {
    $attr_img_src    = wp_get_attachment_image_url( $attachment_id, $size );
    $attr_img_srcset = wp_get_attachment_image_srcset( $attachment_id, $size );
    $attr_sizes      = empty( $sizes ) ? wp_get_attachment_image_sizes( $attachment_id, $size ) : $sizes;

    if ( ! empty( $sizes ) && defined( $sizes ) ) {
        $attr_sizes = constant( $sizes );
    }

    return "<img class='{$class}' src='" . esc_url( $attr_img_src ) . "' sizes='" . esc_attr( $attr_sizes ) . "' srcset='" . esc_attr( $attr_img_srcset ) . "' title='" . get_the_title( $attachment_id ) . "' alt='" . get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) . "' />";
}

/**
 * Animation data attributes
 *
 * @param array $aos
 *
 * @return string|void
 */
function get_the_aos( $aos ) {

    if ( empty( $aos['animation'] ) ) {
        return;
    }

    $data_aos = 'data-aos="' . $aos['animation'] . '"';

    $data_aos .= empty( $aos['easing'] ) ? '' : ' data-aos-easing="' . $aos['easing'] . '"';
    $data_aos .= empty( $aos['delay'] ) ? '' : ' data-aos-delay="' . absint( $aos['delay'] * 1000 ) . '"';
    $data_aos .= empty( $aos['duration'] ) ? '' : ' data-aos-duration="' . absint( $aos['duration'] * 1000 ) . '"';
    $data_aos .= empty( $aos['anchor'] ) ? '' : ' data-aos-anchor="' . $aos['anchor'] . '"';

    return $data_aos;
}

/**
 * Check layout background
 *
 * @param string $background
 *
 * @return bool
 */
function layout_has_bg( $background ) {

    if ( empty( $background ) ) {
        return false;
    }

    if ( 'bg-transparent' !== $background && 'bg-gray-lightest' !== $background ) {
        return true;
    }

    return false;
}
